/* $Id: octave.c,v 1.39 1999/03/10 22:27:01 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Hiller */

/* Functions for Octave option */

#include "sema.h"

#ifdef	MICROSOFT_OS
#  include <io.h>
#  ifdef __STDC__       /* non-ansi names used in this file */
#    define close _close
#    define read _read
#  endif
#else
#  if !defined(E1485_SOURCE) && !defined(HPVXI_DOWNLOAD)
#    include <unistd.h>
#  endif
#endif

#define CHECK(x)			\
    {					\
	err = (x);			\
	if ( err < 0 ) return err;	\
    }
#define ERROR_OUT(e)			\
    {					\
	err = (e);			\
        return i1432_print_error(err);	\
    }

#ifdef MICROSOFT_OS
static char soctbinloc[]="C:\\HPE1432\\LIB\\";
static char soctbinname[]="SOCT.BIN";
static char soctbinname_b[]="SOCTB.BIN";
#else
static char soctbinloc[]="/opt/e1432/lib/";
static char soctbinname[]="soct.bin";
static char soctbinname_b[]="soctb.bin";
#endif

#define MAX_FILE_LEN	120


static const FLOATSIZ32 third_oct_bands[] =
{ 0.8f, 1.0f, 1.25f, 1.6f, 2.0f, 2.5f, 3.15f, 4.0f, 5.0f, 6.3f,
  8.0f, 10.0f, 12.5f };
static const FLOATSIZ32 full_oct_bands[] =
{ 0.5f, 1.0f, 2.0f, 4.0f, 8.0f, 16.0f, 31.5f, 63.0f, 125.0f, 250.0f,
  500.0f, 1000.0f, 2000.0f };

/*ARGSUSED*/
static FLOATSIZ32
oct_band(FLOATSIZ32 freq, SHORTSIZ16 mode)
{
    FLOATSIZ32 base = 1.0;
    FLOATSIZ32 mult, divide;
    const FLOATSIZ32 *table;

    if ( freq < 1.0 ) return 1.0;	/* this is where search begins */
    if ( freq > 1.0e6 ) return 1.0e6;	/* not waste time on large numbers */

    if ( mode == E1432_OCTAVE_MODE_FULL )
    {
        /* full octave repeats in millenimums */
	mult = 1000.0f;
	divide = 0.001f;
	table = &full_oct_bands[0];
        freq *= 1.414213562f; /* sqrt(2), rounding boundary */
    }
    else
    {
        /* third octave repeats in decades */
	mult = 10.0f;
	divide = 0.1f;
	table = &third_oct_bands[0];
        freq *= 1.122462048f; /* sqrt(2^(1/3)), rounding boundary */
    }

    while ( freq > 1.0 )
    {
        base *= mult;
        freq *= divide;
    }
    base *= divide;
    freq *= mult;

    while ( freq > *table) table++;
    table--;

    return *table * base;
}


static FLOATSIZ32
oct_time(FLOATSIZ32 oct_time_val)
{
    FLOATSIZ32 t = 1.0;

    if ( oct_time_val < 0.5e-3 ) return 0.5e-3f;

    if ( oct_time_val < 1.414 )
    /* factors of 2 for 1 second and below */
    {
	while ( oct_time_val < t )
	{
	    t *= .5;
	}
	if ( 2*t/oct_time_val < oct_time_val/t ) return 2*t;
	else return t;
    }
    else
    /* round to seconds */
    {
	return (FLOATSIZ32)((int)(oct_time_val + .5));
    }
}


SHORTSIZ16 EXPORT
e1432_set_octave_avg_mode(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 octave_avg_mode)
{
    TRACE_PRINTF(0, ("e1432_set_avg_mode(0x%p, %d, %d)\n",
      hw, ID, octave_avg_mode));

    if ( octave_avg_mode != E1432_OCTAVE_AVG_MODE_LIN
      && octave_avg_mode != E1432_OCTAVE_AVG_MODE_EXP )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_AVG_MODE);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_AVG_MODE_REG,
      E1432_CMD_SET_OCTAVE_AVG_MODE, (LONGSIZ32) octave_avg_mode);
}


SHORTSIZ16 EXPORT
e1432_set_octave_hold_mode(E1432ID hw, SHORTSIZ16 ID,
  SHORTSIZ16 octave_hold_mode)
{
    TRACE_PRINTF(0, ("e1432_set_hold_mode(0x%p, %d, %d)\n",
      hw, ID, octave_hold_mode));

    if ( octave_hold_mode != E1432_OCTAVE_HOLD_MODE_OFF
      && octave_hold_mode != E1432_OCTAVE_HOLD_MODE_MAX
      && octave_hold_mode != E1432_OCTAVE_HOLD_MODE_MIN )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_HOLD_MODE);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_HOLD_MODE_REG,
      E1432_CMD_SET_OCTAVE_HOLD_MODE, (LONGSIZ32) octave_hold_mode);
}


SHORTSIZ16 EXPORT
e1432_set_octave_start_freq(E1432ID hw, SHORTSIZ16 ID,
  FLOATSIZ32 octave_start_freq)
{
    FLOATSIZ32 band;
    TRACE_PRINTF(0, ("e1432_set_octave_start_freq(0x%p, %d, %g)\n",
      hw, ID, octave_start_freq));

    band = oct_band(octave_start_freq, E1432_OCTAVE_MODE_THIRD);
    if ( band < E1432_OCTAVE_START_FREQ_MIN
      || band > E1432_OCTAVE_START_FREQ_MAX )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_START_FREQ);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_START_FREQ_REG,
      E1432_CMD_SET_OCTAVE_START_FREQ, *(LONGSIZ32*)& band);
}


static SHORTSIZ16
get_octave_freq(E1432ID hw, SHORTSIZ16 ID, LONGSIZ32 reg, FLOATSIZ32 *freq)
{
    FLOATSIZ32 raw_freq;
    SHORTSIZ16 octave_mode;
    SHORTSIZ16 err = 0;

    CHECK(e1432_get_octave_mode(hw, ID, &octave_mode));
    CHECK(i1432_get_mod_float(hw, ID, reg, &raw_freq));

    *freq = oct_band(raw_freq, octave_mode);
    return err;
}


SHORTSIZ16 EXPORT
e1432_get_octave_start_freq(E1432ID hw, SHORTSIZ16 ID,
  FLOATSIZ32 *octave_start_freq)
{
    return get_octave_freq(hw, ID, E1432_OCTAVE_START_FREQ_REG,
      octave_start_freq);
}


SHORTSIZ16 EXPORT
e1432_set_octave_stop_freq(E1432ID hw, SHORTSIZ16 ID,
  FLOATSIZ32 octave_stop_freq)
{
    FLOATSIZ32 band;

    TRACE_PRINTF(0, ("e1432_set_octave_stop_freq(0x%p, %d, %g)\n",
      hw, ID, octave_stop_freq));

    band = oct_band(octave_stop_freq, E1432_OCTAVE_MODE_THIRD);
    if ( band < E1432_OCTAVE_STOP_FREQ_MIN
      || band > E1432_OCTAVE_STOP_FREQ_MAX )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_STOP_FREQ);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_STOP_FREQ_REG,
      E1432_CMD_SET_OCTAVE_STOP_FREQ, *(LONGSIZ32*)& band);
}


SHORTSIZ16 EXPORT
e1432_get_octave_stop_freq(E1432ID hw, SHORTSIZ16 ID,
  FLOATSIZ32 *octave_stop_freq)
{
    return get_octave_freq(hw, ID, E1432_OCTAVE_STOP_FREQ_REG,
      octave_stop_freq);
}


SHORTSIZ16 EXPORT
e1432_set_octave_int_time(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ32 octave_int_time)
{
    FLOATSIZ32 t;

    TRACE_PRINTF(0, ("e1432_set_octave_int_time(0x%p, %d, %g)\n",
      hw, ID, octave_int_time));

    t = oct_time(octave_int_time);
    if ( t < E1432_OCTAVE_INT_TIME_MIN
      || t > E1432_OCTAVE_INT_TIME_MAX )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_INT_TIME);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_INT_TIME_REG,
      E1432_CMD_SET_OCTAVE_INT_TIME, *(LONGSIZ32*)& t);
}


SHORTSIZ16 EXPORT
e1432_set_octave_time_const(E1432ID hw, SHORTSIZ16 ID,
  FLOATSIZ32 octave_time_const)
{
    FLOATSIZ32 t;

    TRACE_PRINTF(0, ("e1432_set_octave_time_const(0x%p, %d, %g)\n",
      hw, ID, octave_time_const));

    t = oct_time(octave_time_const);
    if ( t < E1432_OCTAVE_TIME_CONST_MIN
      || t > E1432_OCTAVE_TIME_CONST_MAX )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_TIME_CONST);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_TIME_CONST_REG,
      E1432_CMD_SET_OCTAVE_TIME_CONST, *(LONGSIZ32*)& t);
}


SHORTSIZ16 EXPORT
e1432_set_octave_time_step(E1432ID hw, SHORTSIZ16 ID,
  FLOATSIZ32 octave_time_step)
{
    FLOATSIZ32 t;

    TRACE_PRINTF(0, ("e1432_set_octave_time_step(0x%p, %d, %g)\n",
      hw, ID, octave_time_step));

    t = oct_time(octave_time_step);
    if ( t < E1432_OCTAVE_TIME_STEP_MIN
      || t > E1432_OCTAVE_TIME_STEP_MAX )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_TIME_STEP);
    }

    return i1432_set_mod(hw, ID, E1432_OCTAVE_TIME_STEP_REG,
      E1432_CMD_SET_OCTAVE_TIME_STEP, *(LONGSIZ32*)& t);
}


SHORTSIZ16 EXPORT
e1432_octave_ctl(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 cmd)
{
    SHORTSIZ16 err = 0;

    TRACE_PRINTF(0, ("e1432_octave_ctl(0x%p, %d, %d)\n", hw, ID, cmd));

    if ( cmd != E1432_OCTAVE_CTL_STOP
      && cmd != E1432_OCTAVE_CTL_RESTART
      && cmd != E1432_OCTAVE_CTL_CONTINUE )
    {
        return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_CTL);
    }

    /* check for valid id */
    CHECK(i1432_checkID(hw, ID));

    /* check if channel or group */
    if ( ID < 0 )
    {
        E1432_MODULE_LIST_NODE *mn;
        E1432_GROUP_LIST_NODE *gn;
        int mod;
	/* signal each module in group */
	gn = i1432_get_group_node(hw, ID);
	for ( mod = 0; mod < gn->modcount; mod++ )
	{
	    mn = gn->modlist[mod];
	    CHECK(e1432_octave_ctl(hw, i1432_get_chan_from_module(mn), cmd));
	}
    }
    else
    {
	/* autozero lone channel */
	CHECK(i1432_write_cmd1(hw, ID, E1432_CMD_OCTAVE_CTL, (LONGSIZ32)cmd));
    }

    return err;
}

 
SHORTSIZ16 EXPORT
e1432_octave_stop(E1432ID hw, SHORTSIZ16 ID)
{
    return e1432_octave_ctl(hw, ID, E1432_OCTAVE_CTL_STOP);
}

 
SHORTSIZ16 EXPORT
e1432_octave_restart(E1432ID hw, SHORTSIZ16 ID)
{
    return e1432_octave_ctl(hw, ID, E1432_OCTAVE_CTL_RESTART);
}

 
SHORTSIZ16 EXPORT
e1432_octave_continue(E1432ID hw, SHORTSIZ16 ID)
{
    return e1432_octave_ctl(hw, ID, E1432_OCTAVE_CTL_CONTINUE);
}


SHORTSIZ16 EXPORT
e1432_set_octave_mode(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 octave_mode)
{
    SHORTSIZ16 err = 0;

    if ( octave_mode != E1432_OCTAVE_MODE_FULL
      && octave_mode != E1432_OCTAVE_MODE_THIRD )
    {
	return i1432_print_error(ERR1432_ILLEGAL_OCTAVE_MODE);
    }

    CHECK(i1432_set_mod(hw, ID, E1432_OCTAVE_MODE_REG,
      E1432_CMD_SET_OCTAVE_MODE, (LONGSIZ32) octave_mode));

    return err;
}


/* Insert one module channel index for channel "ID" into "*reg" and
   increment "reg", ready for next call. */
static SHORTSIZ16
write_chan(E1432ID hw, E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 ID,
  LONGSIZ32 *reg)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 cindex;

    cindex = i1432_get_chan_index(hw, ID);
    CHECK(i1432_direct_write32_register(mn, *reg, cindex));
    *reg += BYTE_SIZE_OF_LONG;

    return err;
}

static SHORTSIZ16
get_curr_data(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 blocksize, int to_double,
  LONGSIZ32 reg, void *ptr)
{
    SHORTSIZ16 err = 0;
    int i;
    if ( to_double )
    {
	FLOATSIZ32 tmp;
	FLOATSIZ64 *dptr = (FLOATSIZ64 *)ptr;
        for ( i = 0; i < blocksize; i++ )
        {
            CHECK(i1432_direct_read32_register(mn, reg, (LONGSIZ32 *)&tmp));
	    *dptr++ = (FLOATSIZ64)tmp;
            reg += BYTE_SIZE_OF_LONG;
        }
    }
    else
    {
	LONGSIZ32 *dptr = (LONGSIZ32 *)ptr;
        for ( i = 0; i < blocksize; i++ )
        {
            CHECK(i1432_direct_read32_register(mn, reg, dptr++));
            reg += BYTE_SIZE_OF_LONG;
        }
    }
    return err;
}


SHORTSIZ16 EXPORT
e1432_get_current_data(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 data_type,
  SHORTSIZ16 data_size, void **data, LONGSIZ32 *actualCnt)
{
    SHORTSIZ16 err = 0;
    E1432_MODULE_LIST_NODE *mn;
    E1432_MODULE_LIST_NODE *last_mn;
    LONGSIZ32 reg = E1432_SCA_BUF;
    const LONGSIZ32 end_chans = -1;
    SHORTSIZ16 meas_type;
    int octave_on;
    LONGSIZ32 blocksize;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    void *dptr;
    int double_data = (data_size == E1432_DATA_SIZE_FLOAT64);

    /* only supports E1432_USER1_DATA (= E1432_OCTAVE_DATA) */
    if ( data_type != E1432_USER1_DATA )
    {
        return i1432_print_error(ERR1432_ILLEGAL_GET_CURRENT_DATA_TYPE);
    }

    /* get mn for first/only channel */
    CHECK(i1432_get_module_from_chan(hw, ID, &mn));

    CHECK(e1432_get_meas_type(hw, ID, &meas_type));
    octave_on = (meas_type == E1432_MEAS_TYPE_OCTAVE);

    if ( octave_on )
    {
	/* only FLOAT32, FLOAT64 supported in Octave */
	if ( data_size != E1432_DATA_SIZE_FLOAT32 && ! double_data )
	{
            return i1432_print_error(ERR1432_ILLEGAL_GET_CURRENT_DATA_SIZE);
	}
    }
    else
    {
	/* only int32 supported in generic user1 */
	if ( data_size != E1432_DATA_SIZE_32 )
	{
            return i1432_print_error(ERR1432_ILLEGAL_GET_CURRENT_DATA_SIZE);
	}
    }

    if ( ID >= 0 )
    {
	/* only one channel, do it separately */
	CHECK(write_chan(hw, mn, ID, &reg));
	/* terminate channel list */
        CHECK(i1432_direct_write32_register(mn, reg, end_chans));
	/* execute the SNAP_CURRENT_DATA command on this module */
        CHECK(i1432_write_cmd1(hw, ID, E1432_CMD_SNAP_CURRENT_DATA, data_type));
    }
    else
    {
        /* Go through modules, writing the channels to be read into the
           E1432_SCA_BUF area and then execute the SNAP_CURRENT_DATA command */
	SHORTSIZ16 lastID;

	/* Iterate thru group, doing first channel manually */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	if ( cn == NULL ) return err;  /* no channels in this group */
	CHECK(i1432_get_module_from_chan(hw, cn->chanID, &mn));
	CHECK(write_chan(hw, mn, cn->chanID, &reg));
	lastID = cn->chanID;
        cn = cn->next;
	last_mn = mn;
	/* Iterate through rest of group */
	while ( cn )
	{
	    CHECK(i1432_get_module_from_chan(hw, cn->chanID, &mn));
	    if ( mn != last_mn )
	    {
		/* terminate channel list */
                CHECK(i1432_direct_write32_register(mn, reg, end_chans));
		/* execute the SNAP_CURRENT_DATA command on the previous
		   module */
                CHECK(i1432_write_cmd1(hw, lastID, E1432_CMD_SNAP_CURRENT_DATA,
		  data_type));
		/* start new module */
		last_mn = mn;
	        reg = E1432_SCA_BUF;
	    }
	    CHECK(write_chan(hw, mn, cn->chanID, &reg));
	    lastID = cn->chanID;
	    cn = cn->next;
	}
	/* finish last module */
        /* terminate channel list */
        CHECK(i1432_direct_write32_register(mn, reg, end_chans));
	/* execute the SNAP_CURRENT_DATA command on the last module */
        CHECK(i1432_write_cmd1(hw, lastID, E1432_CMD_SNAP_CURRENT_DATA,
	  data_type));
    }

    CHECK(i1432_direct_read32_register(mn, E1432_USER1_GCD_LENGTH_REG,
      &blocksize));
    *actualCnt = blocksize;

    if ( ID >= 0 )
    {
        /* do single channel */
	/* data is pointer to the array, rather than an array of pointers */
        dptr = *data;

        CHECK(i1432_direct_read32_register(mn, E1432_USER_CURRENT_BUF, &reg));
        reg *= BYTE_SIZE_OF_LONG;  /* pointer was in 32 bit words */
        CHECK(get_curr_data(mn, blocksize, double_data, reg, dptr));
    }
    else
    {
        gn = i1432_get_group_node(hw, ID);
        cn = gn->chanlist;
	CHECK(i1432_get_module_from_chan(hw, cn->chanID, &mn));
	last_mn = mn;
        CHECK(i1432_direct_read32_register(mn, E1432_USER_CURRENT_BUF, &reg));
        reg *= BYTE_SIZE_OF_LONG;  /* pointer was in 32 bit words */

        while ( cn )
        {
            dptr = *data++;
	    CHECK(i1432_get_module_from_chan(hw, cn->chanID, &mn));
	    if ( mn != last_mn )
	    {
		last_mn = mn;
                CHECK(i1432_direct_read32_register(mn, E1432_USER_CURRENT_BUF,
		  &reg));
                reg *= BYTE_SIZE_OF_LONG;  /* pointer was in 32 bit words */
	    }
            CHECK(get_curr_data(mn, blocksize, double_data, reg, dptr));
            cn = cn->next;
        }
    }

    return err;
}


SHORTSIZ16 EXPORT
e1432_get_current_int32_data(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 data_type,
  LONGSIZ32 **data, LONGSIZ32 *actualCnt)
{
    return e1432_get_current_data(hw, ID, data_type, E1432_DATA_SIZE_32,
     (void **)data, actualCnt);
}


SHORTSIZ16
i1432_octave_download(E1432ID hw, SHORTSIZ16 ID)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 debug_mode;
    int file_des = -1;
    LONGSIZ32 *dptr = NULL;
    SHORTSIZ16 chanID;

    CHECK(i1432_checkID(hw, ID));
    if ( ID < 0 )
    {
	/* get the first channel, and hence module */
	E1432_GROUP_LIST_NODE *gn = i1432_get_group_node(hw, ID);
	chanID = gn->chanlist->chanID;
    }
    else
    {
	/* we have a channel ID already, just use it */
        chanID = ID;
    }
    CHECK(e1432_read32_register(hw, chanID, E1432_OCTAVE_DEBUG_REG,
      &debug_mode));
    /* do download unless debug flagged otherwise */
    if ( ! (debug_mode & E1432_OCTAVE_DEBUG_NO_DOWNLOAD) )
    {
        off_t file_size;
        LONGSIZ32 dsize;
        char file[MAX_FILE_LEN];
	char *binfile;
        LONGSIZ32 sca_id;

        CHECK(e1432_read32_register(hw, chanID, E1432_SINFO_ID_REG, &sca_id));

	binfile =
	  (sca_id == E1432_SCA_ID_SONATA_B) ? soctbinname_b : soctbinname;

        if ( i1432_install_file == NULL )
        {
            (void) strcpy(file, soctbinloc);
        }
        else
        {
            int i;
            (void) strncpy(file, i1432_install_file, MAX_FILE_LEN - 1);
            file[MAX_FILE_LEN - 1] = '\0'; /* insure that terminated */
            /* find end of path and truncate */
            i = strlen(file) - 1;
#ifdef MICROSOFT_OS
            while ( i >= 0 && file[i] != '\\' ) i--;
#else
            while ( i >= 0 && file[i] != '/' ) i--;
#endif
            file[++i] = '\0'; /* truncate */
        }
        /* add file name to string */
        (void) strncat(file, binfile, 120 - 1 - strlen(file));
        err = i1432_open_file(file, &file_des, &file_size);
        if ( err < 0 )
        {
            PRINTF(("problem opening SCA DSP download file %s\n", file));
	    goto cleanup;
        }
        dsize = file_size;
        dptr = (LONGSIZ32 *) malloc(file_size);
        if (dptr == NULL)
        {
            err = i1432_print_error(ERR1432_MALLOC);
	    goto cleanup;
        }
        if (read(file_des, dptr, file_size) != (int) file_size)
        {
            PRINTF(("problem reading SCA DSP download file %s\n", file));
            err = i1432_print_error(ERR1432_FILE);
	    goto cleanup;
        }

#if SWAP
        {
            int     ierror;

            /* Do 4-byte swap.  This is because
	       i1432_sca_dsp_download_mod will swap the data, and we
	       don't want the data swapped because we are downloading
	       an image that already has the correct byte ordering for
	       the module. */
            ierror = ibeswap((char *)dptr, file_size, 4);
            if (ierror != 0)
	    {
                err = i1432_print_error(ERR1432_IO);
	        goto cleanup;
	    }
        }
#endif

        CHECK(i1432_sca_dsp_download_mod(hw, ID, dsize, dptr));
    }

cleanup:
    if ( dptr )
    {
	free(dptr);
    }
    if ( file_des )
    {
	(void) close(file_des);
    }
    return err;
}


SHORTSIZ16
i1432_octave_configure(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 meas_type)
{
    SHORTSIZ16 err;

    if ( meas_type == E1432_MEAS_TYPE_OCTAVE )
    {
        SHORTSIZ16 octave_mode;
        LONGSIZ32 frame_length;

        /* figure out the frame length */
        CHECK(i1432_get_mod_short(hw, ID, E1432_MEAS_TYPE_REG, &octave_mode));
        frame_length = (octave_mode == E1432_OCTAVE_MODE_FULL) ?
          E1432_OCT_FULL_FRAME_LENGTH : E1432_OCT_THIRD_FRAME_LENGTH;

        /* set up the user1 parms */
        CHECK(e1432_set_user_data(hw, ID, frame_length, E1432_OCT_WORD_LENGTH,
         E1432_OCT_SUB_LENGTH, E1432_OCT_SUB_POS));
    }
    else
    {
        /* turn off user_data */
        CHECK(e1432_set_user_data(hw, ID, 0, 0, 0, 0));
    }

    return err;
}


/* Wrapper for partially implemented e1432_set_meas_type() */
SHORTSIZ16 EXPORT
e1432_set_octave_meas(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 meas_type)
{
    return e1432_set_meas_type(hw, ID, meas_type);
}


/* Wrapper for partially implemented e1432_get_meas_type() */
SHORTSIZ16 EXPORT
e1432_get_octave_meas(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 *meas_type)
{
    return e1432_get_meas_type(hw, ID, meas_type);
}
